# R Code for Simulation of Harmonized Compendial UDU Criteria USP <905>
# January 2017
# Lori B. Pfahler (lori.pfahler@merck.com)
# 
# Programmed using the following R Version:
# R version 3.3.2 (2016-10-31) -- "Sincere Pumpkin Patch"
# Copyright (C) 2016 The R Foundation for Statistical Computing
# Platform: x86_64-w64-mingw32/x64 (64-bit)


# =======================================================================
# This function creates the UDU data and evaluates the harmonized 
# UDU test as described in USP <905>. 
# 
# Function Parameters:
# 
# reps =   number of simulated lots for simulation
# Target = target potency (T) at time of manufacture
# mean =   the process mean
# sd =     the process standard deviation
# L1 =     Maximum allowed acceptance value, default is 15
# L2 =     Maximum allowed range for deviation of each dosage unit tested
#          from the calculated value of M in Stage 2, default is 25
# 
# Returns a data.frame of the simulation data results as the 
# second.  Each row in the dataframe is a one simulation run with "reps" 
# being the number of rows in the dataframe.

simUDUTest <- function(reps=1000, Target=100, avg=100, sd=5, L1=15, L2=25)
{	
  # Generate random normal data with mean and sd for appropriate number of
  # lots.  Calculcate average and sd of 10 tablets for stage 1 and average
  # and sd for 30 tablets for stage 2 if needed.
  UDUdata <- matrix(rnorm(30*reps, mean=avg, sd=sd), nrow=reps, ncol=30)
  UDUAvg10 <- rowMeans(UDUdata[,1:10])
  UDUsd10 <- apply(UDUdata[,1:10],1,sd)	
  UDUAvg30 <- rowMeans(UDUdata)
  UDUsd30 <- apply(UDUdata, 1, sd)
  # Create a matrix to hold the results
  results <- matrix(NA, nrow=reps, ncol=14, dimnames=list(NULL, 
    c("UDUAvg10", "UDUsd10", "AV10", "MS1", "Pass.S1", "UDUAvg30", "UDUsd30", "AV30",
    "MS2", "LL", "UL", "nOut", "Pass.S2", "PassOverall")))
  # Determine M for stage 1
  if(Target<= 101.5) 
  {
    MS1 <- ifelse(UDUAvg10<=98.5,98.5,ifelse(UDUAvg10>=101.5,101.5,UDUAvg10))
  }
  if(Target>101.5)
  {
    MS1 <- ifelse(UDUAvg10<=98.5,98.5,ifelse(UDUAvg10>=Target,Target,UDUAvg10))
  }	
  # Calculate Acceptance Value for Stage 1
  AV10tablets <- abs(MS1-UDUAvg10)+2.4*UDUsd10
  # Flag for Pass of Stage 1: 1=pass, 0=fail
  PassS1 <- ifelse(AV10tablets<=L1, 1, 0)
  # Determine M for stage 2
  if(Target<= 101.5) 
  {
    MS2 <- ifelse(UDUAvg30<=98.5,98.5,ifelse(UDUAvg30>=101.5,101.5,UDUAvg30))
  }
  if(Target>101.5)
  {
    MS2 <- ifelse(UDUAvg30<=98.5,98.5,ifelse(UDUAvg30>=Target,Target,UDUAvg30))
  }	
  # Calculate Acceptance Value for Stage 2
  AV30tablets <- abs(MS2-UDUAvg30)+2.0*UDUsd30
  # Calculate number of results outside of 75 to 125% for Stage 2
  LowerLimit <- (1-(0.01*L2))*MS2
  UpperLimit <- (1+(0.01*L2))*MS2
  CountOutside <- rowSums(UDUdata<LowerLimit)+rowSums(UDUdata>UpperLimit)
  # Flag for Pass of Stage 2: 1=pass, 0=fail
  PassS2 <- ifelse((CountOutside==0 & AV30tablets<=L1), 1, 0)
  # Flag for Overall Pass - ifelse statement controls if Stage 2 
  # result is considered:  1=pass, 0=fail
  PassOverall <- ifelse(PassS1==1,1,ifelse(PassS2==1,1,0))
  # Place results in results matrix
  results[,1] <- UDUAvg10
  results[,2] <- UDUsd10
  results[,3] <- AV10tablets
  results[,4] <- MS1
  results[,5] <- PassS1
  results[,6] <- UDUAvg30
  results[,7] <- UDUsd30
  results[,8] <- AV30tablets
  results[,9] <- MS2
  results[,10] <- LowerLimit
  results[,11] <- UpperLimit
  results[,12] <- CountOutside
  results[,13] <- PassS2
  results[,14] <- PassOverall
  return(data.frame(results))	
}

# Test the function
test1 <- simUDUTest(reps=10, T=100, avg=100, sd=5)
test2 <- simUDUTest(reps=30, T=100, avg=100, sd=10)

# Example of running simulation for a range of standard deviation with
# with avg  100%LC - Useful for making Operating Characteristic Curves
UDU100 <- matrix(data=NA, nrow=length(seq(1,10,0.1)), ncol=3, dimnames = list(NULL, c("Avg", "SD", "PassUSP")))
for (i in seq(1, 10, 0.1))
{
  j <- ifelse(i==1, 1, j+1)
  runCurrent <- simUDUTest(reps=5000, Target=100, avg=100, sd=i, L1=15, L2=25)
  UDU100[j,1] <- 100
  UDU100[j,2] <- i
  UDU100[j,3] <- mean(runCurrent$PassOverall)
}

# Create OC Curve with Data from example above
plot(UDU100[,2], UDU100[,3], xlab="Standard Deviation (%LC)", 
  ylab="Probability to Accept", type="l", ylim=c(0,1))

